<?php
// Include the database connection
include '../dbconn.php';

// Allow CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

// Check if the request method is POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Retrieve form data
    $mobile = $_POST['mobile'] ?? null;
    $otp = $_POST['otp'] ?? null;
    $new_password = $_POST['new_password'] ?? null;
    $confirm_new_password = $_POST['confirm_new_password'] ?? null;

    // Initialize the response
    $response = [
        "status" => "error",
        "message" => "Failed to reset password. Please try again later."
    ];

    // Validate inputs
    if (!$mobile || !preg_match('/^\d{10}$/', $mobile)) {
        $response['message'] = "Invalid mobile number format. Please provide a valid 10-digit mobile number.";
        echo json_encode($response);
        exit;
    }

    if (!$otp || !preg_match('/^\d{4}$/', $otp)) {
        $response['message'] = "Invalid OTP format. Please provide a valid 4-digit OTP.";
        echo json_encode($response);
        exit;
    }

    if (!$new_password || !$confirm_new_password) {
        $response['message'] = "Password fields cannot be empty.";
        echo json_encode($response);
        exit;
    }

    if ($new_password !== $confirm_new_password) {
        $response['message'] = "New Password and Confirm New Password do not match.";
        echo json_encode($response);
        exit;
    }

    // Allow for testing with a fixed OTP
    if ($otp === "2000") {
        // If OTP is 2000, skip database validation
        $otpValid = true;
    } else {
        // Check if the OTP is valid in the database
        $otpQuery = "SELECT otp FROM otp_store WHERE mobile = ?";
        $otpStmt = $conn->prepare($otpQuery);
        $otpStmt->bind_param("s", $mobile);
        $otpStmt->execute();
        $otpResult = $otpStmt->get_result();
        
        if ($otpResult->num_rows === 0) {
            $response['message'] = "No OTP record found for the given mobile number.";
            echo json_encode($response);
            exit;
        }
        
        $otpRow = $otpResult->fetch_assoc();
        $storedOtp = trim($otpRow['otp']); // Trim any extra spaces

        $otpValid = ($storedOtp === $otp);
    }

    if (!$otpValid) {
        $response['message'] = "Invalid OTP.";
        echo json_encode($response);
        exit;
    }

    // Hash the new password
    $hashedPassword = password_hash($new_password, PASSWORD_BCRYPT);

    // Update the user's password
    $updateQuery = "UPDATE users SET password_hash = ? WHERE mobile = ?";
    $updateStmt = $conn->prepare($updateQuery);
    $updateStmt->bind_param("ss", $hashedPassword, $mobile);
    $updateSuccess = $updateStmt->execute();

    if ($updateSuccess) {
        // Delete the OTP from the OTP table after successful password reset
        $deleteOtpQuery = "DELETE FROM otp_store WHERE mobile = ?";
        $deleteOtpStmt = $conn->prepare($deleteOtpQuery);
        $deleteOtpStmt->bind_param("s", $mobile);
        $deleteOtpStmt->execute();

        $response = [
            "status" => "success",
            "message" => "Password reset successfully."
        ];
    }

    // Output the response
    echo json_encode($response);
    $otpStmt->close();
    $conn->close();
} else {
    // Return error if the request method is not POST
    echo json_encode([
        "status" => "error",
        "message" => "Invalid request method. Use POST."
    ]);
}
?>
