<?php
header("Content-Type: application/json");
require_once '../dbconn.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(["success" => false, "error" => "Invalid request method"]);
    exit;
}

// Validate input fields
if (!isset($_POST['product_id'], $_POST['quantity_changed'], $_POST['change_type'])) {
    echo json_encode(["success" => false, "error" => "Missing required fields"]);
    exit;
}

$product_id = intval($_POST['product_id']);
$quantity_changed = intval($_POST['quantity_changed']);
$change_type = $_POST['change_type'];
$notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

// Validate input values
if ($product_id <= 0 || $quantity_changed <= 0 || !in_array($change_type, ['addition', 'subtraction'])) {
    echo json_encode(["success" => false, "error" => "Invalid input values"]);
    exit;
}

// Start a transaction
$conn->begin_transaction();

try {
    // Get current stock quantity
    $query = "SELECT stock_quantity FROM products WHERE id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $stmt->bind_result($current_stock);
    $stmt->fetch();
    $stmt->close();

    if ($current_stock === null) {
        throw new Exception("Product not found");
    }

    // Calculate new stock quantity
    $new_stock_quantity = ($change_type === 'addition') ? 
        ($current_stock + $quantity_changed) : 
        ($current_stock - $quantity_changed);

    if ($new_stock_quantity < 0) {
        throw new Exception("Not enough stock available");
    }

    // Update stock in products table
    $update_query = "UPDATE products SET stock_quantity = ?, updated_at = NOW() WHERE id = ?";
    $stmt = $conn->prepare($update_query);
    $stmt->bind_param("ii", $new_stock_quantity, $product_id);
    $stmt->execute();
    $stmt->close();

    // Insert into inventory history
    $history_query = "INSERT INTO inventory_history (product_id, quantity_changed, stock_after_change, change_type, notes, created_at) 
                      VALUES (?, ?, ?, ?, ?, NOW())";
    $stmt = $conn->prepare($history_query);
    $stmt->bind_param("iiiss", $product_id, $quantity_changed, $new_stock_quantity, $change_type, $notes);
    $stmt->execute();
    $stmt->close();

    // Commit transaction
    $conn->commit();

    echo json_encode([
        "success" => true,
        "product_id" => $product_id,
        "new_stock_quantity" => $new_stock_quantity,
        "updated_at" => date('Y-m-d H:i:s')
    ]);

} catch (Exception $e) {
    // Rollback transaction if an error occurs
    $conn->rollback();
    echo json_encode(["success" => false, "error" => $e->getMessage()]);
}

// Close database connection
$conn->close();
?>
